`timescale 1ns/1ps

module testbench;
//-----------------------------------------------------------------------------
// Enable verification summary for console
// Warning! Verification may slow down simulation.
`define __VER
//-----------------------------------------------------------------------------
// Defines SPI operation mode to be used: if uncommented then mode 0 will be
// used, otherwise will be used mode 3.
// Notice: difference between modes - in mode 0 inactive state of SCK signal is
//   "0", while in mode 3 inactive state of SCK signal is "1".
`define __MODE0
//-----------------------------------------------------------------------------

//========================================================================
// Array constants table  
//========================================================================
parameter numWLP  = 8;
parameter numWL   = 1024;
parameter numMux  = 256;
parameter numSect = 8;

parameter D_MSB = 8;
parameter A_MSB = 21;
parameter A_SEC_MSB = 18;

parameter sizePage   = numWLP * numMux;
parameter sizeSector = numWL * numMux;
parameter sizeMem    = numSect * sizeSector;

//========================================================================
// Timing units table  
//========================================================================
parameter ns_unit =  1;
parameter us_unit =  1000;
parameter ms_unit =  1000 * 1000;
parameter sec_unit = 1000 * 1000 * 1000;

//========================================================================
// Timing table for SPI I/F  
//========================================================================
parameter	Tvcs = 100_000;	// Ucc setup time in ns

parameter Twh_sck_min = 10; // SCK pulse high time, ns
parameter Twl_sck_min = 10; // SCK pulse low time, ns

parameter Twh_sck_rd1 = 17; // SCK pulse high time, ns
parameter Twl_sck_rd1 = 17; // SCK pulse low time, ns

parameter Twh_sck_rd2 = 34; // SCK pulse high time, ns
parameter Twl_sck_rd2 = 34; // SCK pulse low time, ns

parameter Tsu1_ce = 10; // CS setup time for "0", ns
parameter Th1_ce = 5; // CS hold time for "1", ns

parameter Twh1_ce = 1000;
parameter Twh2_ce = 50;

parameter Tsu_sck_si = 2; // SDI to SCK setup time, ns
parameter Th_si_sck = 1; // SDI to SCK hold time, ns

parameter Tpzd_sck_d = 10;
parameter Tpdz_sck_d = 10;

integer i, j, k;

//========================================================================
// Operation codes for SPI I/F  
//========================================================================
parameter OP_READ1      = 8'h0B;
parameter OP_READ2      = 8'h03;
parameter OP_WR1        = 8'h02;
parameter OP_WR2        = 8'hB2;
parameter OP_ER1        = 8'hD8;
parameter OP_ER2        = 8'h60;
parameter OP_WEL        = 8'h06;
parameter OP_WDL        = 8'h04;
parameter OP_Protect    = 8'h36;
parameter OP_Unprotect  = 8'h39;
parameter OP_RDStatus   = 8'h05;
parameter OP_WRStatus   = 8'h01;
parameter OP_RDPrReg    = 8'h3C;
parameter OP_RDID       = 8'h9F;
parameter OP_Reset      = 8'hF0;

//------------------------------------------------------------------------
// I/O Ports                                                            
//------------------------------------------------------------------------
reg  nCE; 
reg  SCK; 
reg  SDI; 
tri  SDO; 

//------------------------------------------------------------------------
// Testbench routines                                                   
//------------------------------------------------------------------------
reg  [7:0] rdbuf;  // read for dut buffer
reg  [7:0] rdbufm; // read for ethalon buffer
reg  [7:0] wrbuf;	// write buffer
reg  [7:0] rdstatusbuf;	// read buffer for status
reg  [7:0] rdidbuf [0:1];	// read buffer for id
reg  [7:0] opcode; // current command opcode

reg  [A_MSB-1:0]  ap; // address pointer
reg               match; // match flag
reg               dismatch; // dismatch flag

integer test_no;

reg [7:0] mem [sizeMem-1:0]; // ethalon memory

//=============================================================================
//===== TOP Device =====
//=============================================================================
K1636RR4U dut (
  .nWE(1'b1),
  .nOE(1'b1),
  .nCE(nCE),
  .A({A_MSB{1'b0}}),
  .D(),
  .A9_HV(),
  .OE_HV(),
  .TDI(),
  .TCK(),
  .STROBE(),
  .MRST(),
  .SCK(SCK),
  .SI(SDI),
  .SEL_SPI(1'b1),
  .SO(SDO)
); 

//=============================================================================
//===== Initializing ports  =====
//=============================================================================
initial begin
  nCE = 1'b1;
 `ifdef __MODE0
  SCK = 1'b0;
  `else
  SCK = 1'b1;
  `endif
  SDI = 1'bz;

  ap = {A_MSB{1'b0}};
  match = 1'b1;
  dismatch = 1'b0;
end

//=============================================================================
//===== Test sequence itself =====
//=============================================================================
initial begin
  #(Tvcs); // Wait for model wakeup

  $display("RUN[TEST]: SPI I/F testing");

//=== Test1 - Autoselect ===========================================
  test_no=1;  

  CMD_ReadID;

//=== Test2 - Sector Protection ====================================
  test_no=2;  
  ap = {3'h0, {A_SEC_MSB{1'b0}}};

  for (j=0; j<numSect; j=j+1) CMD_RdPrSect(j); // Read Sector Protect Status

  for (j=0; j<numSect; j=j+1) begin // Unprotect All Sectors
    CMD_WriteEnable;
    CMD_UnProtSect(j);
  end

  for (j=0; j<numSect; j=j+1) CMD_RdPrSect(j); // Read Sector Protect Status

  for (j=4; j<numSect; j=j+1) begin // Protect Sectors from 4th to 8th
    CMD_WriteEnable;
    CMD_ProtSect(j);
  end

  for (j=0; j<numSect; j=j+1) CMD_RdPrSect(j); // Read Sector Protect Status

  for (j=4; j<numSect; j=j+1) begin // Protect Sectors from 4th to 8th
    CMD_WriteEnable;
    CMD_UnProtSect(j);
  end

 for (j=0; j<numSect; j=j+1) CMD_RdPrSect(j); // Read Sector Protect Status

//=== Test3 - Status register ======================================
  test_no=3; 
  ap = {3'h0, {A_SEC_MSB{1'b0}}};

  CMD_ReadStatus(1);  
  CMD_WriteEnable;
  CMD_WriteStatus(8'b01000000); // set RSTE bit to "1"
  CMD_ReadStatus(1);

//=== Test4 - Erase chip ===========================================
  test_no=4; 
  ap = {3'h0, {A_SEC_MSB{1'b0}}};

  CMD_WriteEnable; 
  fork   
    CMD_EraseChip; 
    //#(100 * ms_unit) CMD_Reset; 
  join

  ap = {3'h0, {A_SEC_MSB{1'b0}}};

  // Verify erase chip
  CMD_Read(ap, numMux, 1); // Check with dummy signle word line 
                           // words from sector 1

//=== Test5 - Write random =========================================
  test_no=5; 
  ap = {3'h0, {A_SEC_MSB{1'b0}}};

  for (j=0; j<4; j=j+1) begin
    for (k=0; k<numMux; k=k+1) begin 
      FillWrBuffer;
      CMD_WriteEnable;     
      fork
        CMD_Write(ap, wrbuf);    
        //#(10 * us_unit) CMD_Reset;
      join
      ap[A_SEC_MSB-1:0]=ap[A_SEC_MSB-1:0]+1;
    end
    ap[A_SEC_MSB-1:0] = {A_SEC_MSB{1'b0}};
    ap[A_MSB-1:A_SEC_MSB]=ap[A_MSB-1:A_SEC_MSB]+1;
  end

  ap = {3'h0, {A_SEC_MSB{1'b0}}};

  // Verify write random
  for (j=0; j<3; j=j+1) begin
    CMD_Read(ap, 64, 1); // Check with dummy 64 words from sectors 1-3
    CMD_Read(ap, 64, 0); // Check without dummy 64 words from sectors 1-3
  
    ap[A_SEC_MSB-1:0] = {A_SEC_MSB{1'b0}};
    ap[A_MSB-1:A_SEC_MSB]=ap[A_MSB-1:A_SEC_MSB]+1;    
  end

//=== Test6 - Erase sector ==========================================
  test_no=6; 
  ap = {3'h0, {A_SEC_MSB{1'b0}}};

  for (j=0; j<2; j=j+1) begin
    CMD_WriteEnable;  
    fork
      CMD_EraseSector(ap[A_MSB-1:A_SEC_MSB]);
      //#(10 * ms_unit) CMD_Reset;
    join
    ap[A_MSB-1:A_SEC_MSB]=ap[A_MSB-1:A_SEC_MSB]+1;
  end

  // Verify erase sector
  for (j=0; j<3; j=j+1) begin
    CMD_Read(ap, 64, 1); // Check with dummy 64 words from sectors 1-3
  
    ap[A_SEC_MSB-1:0] = {A_SEC_MSB{1'b0}};
    ap[A_MSB-1:A_SEC_MSB]=ap[A_MSB-1:A_SEC_MSB]+1;    
  end

//===================================================================
  `ifdef __VER
  if (dismatch) $display("ERROR[TEST]: Some operations unsuccessfully finished or read data is not matched with ethalon!");
  else $display("DONE[TEST]: All operations successfully finished and data is matched with ethalon");
  `else
  $display("DONE[TEST]: All operations successfully finished");
  `endif
  
  $display("\n---< See ya later, allagator! >---\n");

  $finish;
end

always @ (negedge match) dismatch = 1; // dismatch test detector

//=============================================================================
//===== TASKS =====
//=============================================================================
// Fills write buffer with random numbers.
task FillWrBuffer;
begin
  wrbuf=$random;
end
endtask


// Check status of internal operation
task CheckStatus;
input time period;
output epe;

reg[31:0] N;
integer i;
begin
  N=1;
  epe = 0;
  
  for (i=0; i<N; i=i+1) begin // check end of write cycle
    CMD_ReadStatus(1);
    if (rdstatusbuf[5]) begin 
      N=0; 
      epe = 1;
      `ifdef __VER
      $display("ERROR[Internal operation]: EPE bit is detected "); 
      `endif
    end else if (rdstatusbuf[0]) begin
      N=N+1;
      #(period);
    end
  end 

end
endtask

//=============================================================================
//===== SPI routines =====
//=============================================================================

// Sends one data byte through SPI.
task SPISendByte;
input [7:0] data;   // data byte
input start;
input stop;

input time Twl_sck_i;
input time Twh_sck_i;

integer i;
begin
  // data send with MSB first!
  if (start) begin
    `ifdef __MODE0
    fork
      #(Twh2_ce/2) nCE = 1'b0;
      #(Twh2_ce/2 + Tsu1_ce - Tsu_sck_si) SDI = data[7];
      #(Twh2_ce/2 + Tsu1_ce) SCK = 1'b1;
      //#(Twh2_ce/2 + Tsu1_ce + Th_si_sck) SDI = 1'bx;
      #(Twh2_ce/2 + Tsu1_ce + Twh_sck_i) SCK = 1'b0;
    join
    `else
    fork
      #(Twh2_ce/2) nCE = 1'b0;
      #(Twh2_ce/2 + Th2_ce) SCK = 1'b0;
      #(Twh2_ce/2 + Th2_ce + Twl_sck_i - Tsu_sck_si) SDI = data[7];
      #(Twh2_ce/2 + Th2_ce + Twl_sck_i) SCK = 1'b1;
      //#(Twh2_ce/2 + Th2_ce + Twl_sck_i + Th_si_sck) SDI = 1'bx;
      #(Twh2_ce/2 + Th2_ce + Twl_sck_i + Twh_sck_i) SCK = 1'b0;
    join
    `endif
  end else begin
    fork
      #(Twl_sck_i - Tsu_sck_si) SDI = data[7];
      #(Twl_sck_i) SCK = 1'b1;
      //#(Twl_sck_i + Th_si_sck) SDI = 1'bx;
      #(Twl_sck_i + Twh_sck_i) SCK = 1'b0;
    join
  end

  for (i = 6; i >= 1; i = i - 1) begin
    fork
      #(Twl_sck_i - Tsu_sck_si) SDI = data[i];
      #(Twl_sck_i) SCK = 1'b1;
      //#(Twl_sck_i + Th_si_sck) SDI = 1'bx;
      #(Twl_sck_i + Twh_sck_i) SCK = 1'b0;
    join
  end

  if (stop) begin
  `ifdef __MODE0
    fork
      #(Twl_sck_i - Tsu_sck_si) SDI = data[0];
      #(Twl_sck_i) SCK = 1'b1;
      //#(Twl_sck_i + Th_si_sck) SDI = 1'bx;
      #(Twl_sck_i + Th1_ce) nCE = 1'b1;
      #(Twl_sck_i + Th1_ce + Twh2_ce/2);
      #(Twl_sck_i + Twh_sck_i) SCK = 1'b0;
    join
  `else
    fork
      #(Twl_sck_i - Tsu_sck_si) SDI = data[0];
      #(Twl_sck_i) SCK = 1'b1;
      //#(Twl_sck_i + Th_si_sck) SDI = 1'bx;
      #(Twl_sck_i + Th1_ce) nCE = 1'b1;
      #(Twl_sck_i + Th1_ce + Twh2_ce/2);
      #(Twl_sck_i + Twh_sck_i);
    join
  `endif
  end else begin
    fork
      #(Twl_sck_i - Tsu_sck_si) SDI = data[0];
      #(Twl_sck_i) SCK = 1'b1;
      //#(Twl_sck_i + Th_si_sck) SDI = 1'bx;
      #(Twl_sck_i + Twh_sck_i) SCK = 1'b0;
    join
  end
end
endtask

// Reads one data byte via SPI.
task SPIReadByte;
output [7:0] data;  // read data byte
input stop;

input time Twl_sck_i;
input time Twh_sck_i;

integer i;

begin
  data = 8'h00;

  // data send with MSB first!
  for (i = 7; i >= 1; i = i - 1) begin
    fork
      #(Tpzd_sck_d + 0.1) data[i] = SDO;
      #(Twl_sck_i) SCK = 1'b1;
      #(Twl_sck_i + Twh_sck_i) SCK = 1'b0;
    join
  end

  if (stop) begin
  `ifdef __MODE0
    fork
      #(Tpzd_sck_d + 0.1) data[0] = SDO;
      #(Twl_sck_i) SCK = 1'b1;
      #(Twl_sck_i + Twh_sck_i) SCK = 1'b0;
      #(Twl_sck_i + Twh_sck_i) nCE = 1'b1;
      #(Twl_sck_i + Twh_sck_i + Twh2_ce/2);
    join
  `else
    fork
      #(Tpzd_sck_d + 0.1) data[0] = SDO;
      #(Twl_sck_i) SCK = 1'b1;
      #(Twl_sck_i + Th1_ce) nCE = 1'b1;
      #(Twl_sck_i + Twh_sck_i + Twh2_ce/2);
      #(Twl_sck_i + Twh_sck_i);
    join
  `endif
  end else begin
    fork
      #(Tpzd_sck_d + 0.1) data[0] = SDO;
      #(Twl_sck_i) SCK = 1'b1;
      #(Twl_sck_i + Twh_sck_i) SCK = 1'b0;
    join
  end
end
endtask

//=============================================================================
//===== SPI commands =====
//=============================================================================
// 1,2 Reads N bytes via SPI (has dummy byte before read started). 
task CMD_Read;
input [A_MSB-1:0] addr; // address to read
input [31:0] rdbytes;   // number of bytes to read
input dummy; // with dummy byte

integer i;
reg   [31:0] rdb;   // actual number of words to read
time Twl_sck_i;
time Twh_sck_i;

begin
  match = 1'b1;
  rdb   = rdbytes;
  if (dummy) begin
    opcode = OP_READ1;
    Twl_sck_i = Twl_sck_rd1;
    Twh_sck_i = Twh_sck_rd1;
  end else begin
    opcode = OP_READ2;
    Twl_sck_i = Twl_sck_rd2;
    Twh_sck_i = Twh_sck_rd2;
  end  

  SPISendByte(opcode, 1, 0, Twl_sck_i, Twh_sck_i);
  SPISendByte({{(8-(A_MSB-16)){1'b1}}, addr[A_MSB-1:16]}, 0, 0, Twl_sck_i, Twh_sck_i);
  SPISendByte(addr[15:8], 0, 0, Twl_sck_i, Twh_sck_i);
  SPISendByte(addr[7:0], 0, 0, Twl_sck_i, Twh_sck_i);
  if (dummy) SPISendByte(8'hFF, 0, 0, Twl_sck_i, Twh_sck_i);    // dummy
  for (i = 0; i < rdb; i = i + 1) begin
    if (i == rdb - 1) SPIReadByte(rdbuf, 1, Twl_sck_i, Twh_sck_i);
    else SPIReadByte(rdbuf, 0, Twl_sck_i, Twh_sck_i);

    `ifdef __VER
    rdbufm=mem[i+addr];
    if (rdbuf==rdbufm) begin
      match=1'b1;
      //$display("DONE[SCH vs Ethalone]: rdbuf(%h)==rdbufm(%h), MATCH DATA =)",rdbuf,rdbufm); 
    end else begin
      match=1'b0;
      $display("ERROR[SCH vs Ethalone]: rdbuf(%h)!=rdbufm(%h), addr=%h, DISMATCH DATA =( ",rdbuf,rdbufm,(i+addr));
    end
    `endif 
  end   
end
endtask

// 3 Writes N data bytes via SPI.
task CMD_Write;
input [A_MSB-1:0] addr;  // address to write
input [7:0] data;  // data to write

reg epe;

time Twl_sck_i;
time Twh_sck_i;

begin 
   opcode = OP_WR1;
   Twl_sck_i = Twl_sck_min;
   Twh_sck_i = Twh_sck_min;

   SPISendByte(opcode, 1, 0, Twl_sck_i, Twh_sck_i);
   SPISendByte({{(8-(A_MSB-16)){1'b1}}, addr[A_MSB-1:16]}, 0, 0, Twl_sck_i, Twh_sck_i);
   SPISendByte(addr[15:8], 0, 0, Twl_sck_i, Twh_sck_i);
   SPISendByte(addr[7:0], 0, 0, Twl_sck_i, Twh_sck_i);
   SPISendByte(data, 0, 1, Twl_sck_i, Twh_sck_i);

   `ifdef __VER
   $display("RUN[Program]: addr=%h, data=%h", addr, data); 
   `endif 

   #(Twh1_ce - Twh2_ce) CheckStatus(50 * us_unit, epe);

   `ifdef __VER
   if (epe) begin
      $display("ERROR[Program]: Operation Fail =("); 
   end else begin
      $display("DONE[Program]: Operation Successful =)"); 
   end
   mem[addr]=data;
   `endif  
end
endtask

// 5 Erases sector data via SPI.
task CMD_EraseSector;
input [A_MSB-A_SEC_MSB-1:0] addr_sec;  // address of sector to erase

reg epe;

time Twl_sck_i;
time Twh_sck_i;

begin
   opcode = OP_ER1;
   Twl_sck_i = Twl_sck_min;
   Twh_sck_i = Twh_sck_min;

   SPISendByte(opcode, 1, 0, Twl_sck_i, Twh_sck_i);
   SPISendByte({2'b00, addr_sec, 2'b00}, 0, 0, Twl_sck_i, Twh_sck_i);
   SPISendByte(8'h00, 0, 0, Twl_sck_i, Twh_sck_i);
   SPISendByte(8'h00, 0, 1, Twl_sck_i, Twh_sck_i);

   `ifdef __VER
   $display("RUN[Erase Sector]: Sector %d", addr_sec);
   `endif  

   #(Twh1_ce - Twh2_ce) CheckStatus(50 * ms_unit, epe);

   `ifdef __VER
   if (epe) begin
      $display("ERROR[Erase Sector]: Operation Fail =("); 
   end else begin
      $display("DONE[Erase Sector]: Operation Successful =)"); 
   end
   for (i=0; i<sizeSector; i=i+1) mem[{addr_sec,{A_SEC_MSB{1'b0}}}+i]<={D_MSB{1'b1}};
   `endif   
end
endtask

// 6 Erases full chip via SPI.
task CMD_EraseChip;
reg epe;

time Twl_sck_i;
time Twh_sck_i;
begin
   opcode = OP_ER2;
   Twl_sck_i = Twl_sck_min;
   Twh_sck_i = Twh_sck_min;

   SPISendByte(opcode, 1, 1, Twl_sck_i, Twh_sck_i);

   `ifdef __VER
   $display("RUN[Erase Chip]");
   `endif  

   #(Twh1_ce - Twh2_ce) CheckStatus(100 * ms_unit, epe);

   `ifdef __VER
   if (epe) begin
      $display("ERROR[Erase Chip]: Operation Fail =("); 
   end else begin
      $display("DONE[Erase Chip]: Operation Successful =)"); 
   end
   for (i=0; i<sizeMem; i=i+1) mem[i]<={D_MSB{1'b1}};
   `endif   

end
endtask

// 7 Sets WEL bit to "1" via SPI.
task CMD_WriteEnable;
time Twl_sck_i;
time Twh_sck_i;
begin
   opcode = OP_WEL;
   Twl_sck_i = Twl_sck_min;
   Twh_sck_i = Twh_sck_min;

   SPISendByte(opcode, 1, 1, Twl_sck_i, Twh_sck_i);
   #(Twh1_ce - Twh2_ce);
end
endtask

// 8 Sets WEL bit to "0" via SPI.
task CMD_WriteDisable;
time Twl_sck_i;
time Twh_sck_i;
begin
   opcode = OP_WDL;
   Twl_sck_i = Twl_sck_min;
   Twh_sck_i = Twh_sck_min;

   SPISendByte(opcode, 1, 1, Twl_sck_i, Twh_sck_i);
   #(Twh1_ce - Twh2_ce);
end
endtask

// 9 Protect sector for WR/ER operation via SPI.
task CMD_ProtSect;
input [A_MSB-A_SEC_MSB-1:0] addr;  // address of sector

time Twl_sck_i;
time Twh_sck_i;
begin
   opcode = OP_Protect;
   Twl_sck_i = Twl_sck_min;
   Twh_sck_i = Twh_sck_min;

   SPISendByte(opcode, 1, 0, Twl_sck_i, Twh_sck_i);
   SPISendByte({3'b000, addr, 2'b00}, 0, 0, Twl_sck_i, Twh_sck_i);
   SPISendByte(8'hFF, 0, 0, Twl_sck_i, Twh_sck_i);
   SPISendByte(8'hFF, 0, 1, Twl_sck_i, Twh_sck_i);
   #(Twh1_ce - Twh2_ce);
end
endtask

// 10 Disable protect sector for WR/ER operation via SPI.
task CMD_UnProtSect;
input [A_MSB-A_SEC_MSB-1:0] addr;  // address of sector

time Twl_sck_i;
time Twh_sck_i;
begin
   opcode = OP_Unprotect;
   Twl_sck_i = Twl_sck_min;
   Twh_sck_i = Twh_sck_min;

   SPISendByte(opcode, 1, 0, Twl_sck_i, Twh_sck_i);
   SPISendByte({3'b000, addr, 2'b00}, 0, 0, Twl_sck_i, Twh_sck_i);
   SPISendByte(8'hFF, 0, 0, Twl_sck_i, Twh_sck_i);
   SPISendByte(8'hFF, 0, 1, Twl_sck_i, Twh_sck_i);
   #(Twh1_ce - Twh2_ce);
end
endtask

// 11 Read protect sector register via SPI.
task CMD_RdPrSect;
input [A_MSB-A_SEC_MSB-1:0] addr;  // address of sector

time Twl_sck_i;
time Twh_sck_i;
begin
   opcode = OP_RDPrReg;
   Twl_sck_i = Twl_sck_rd1;
   Twh_sck_i = Twh_sck_rd1;

   SPISendByte(opcode, 1, 0, Twl_sck_i, Twh_sck_i);
   SPISendByte({3'b000, addr, 2'b00}, 0, 0, Twl_sck_i, Twh_sck_i);
   SPISendByte(8'hFF, 0, 0, Twl_sck_i, Twh_sck_i);
   SPISendByte(8'hFF, 0, 0, Twl_sck_i, Twh_sck_i);
   SPIReadByte(rdbuf[7:0], 1, Twl_sck_i, Twh_sck_i);

   `ifdef __VER
   if (rdbuf[7:0]==8'hFF) $display("Verify[Protect Sector]: Sector %d is protected for erase/program operations", addr);
   else $display("Verify[Protect Sector]: Sector %d is unprotected for erase/program operations", addr);
   `endif
end
endtask

// 12 Write status register via SPI.
task CMD_WriteStatus;
input [7:0] data;  // data to write

time Twl_sck_i;
time Twh_sck_i;
begin 
   opcode = OP_WRStatus;
   Twl_sck_i = Twl_sck_min;
   Twh_sck_i = Twh_sck_min;

   SPISendByte(opcode, 1, 0, Twl_sck_i, Twh_sck_i);
   SPISendByte(data, 0, 1, Twl_sck_i, Twh_sck_i);
   #(Twh1_ce - Twh2_ce);
end
endtask

// 13 Read status register via SPI.
task CMD_ReadStatus;
input [31:0] N;

time Twl_sck_i;
time Twh_sck_i;
integer i;
begin
   opcode = OP_RDStatus;
   Twl_sck_i = Twl_sck_rd1;
   Twh_sck_i = Twh_sck_rd1;

   SPISendByte(opcode, 1, 0, Twl_sck_i, Twh_sck_i);
   for (i = 0; i < (N-1); i = i + 1) begin
     SPIReadByte(rdstatusbuf, 0, Twl_sck_i, Twh_sck_i);
   end
   SPIReadByte(rdstatusbuf, 1, Twl_sck_i, Twh_sck_i);
end
endtask

// 14 Read ID via SPI.
task CMD_ReadID;
time Twl_sck_i;
time Twh_sck_i;
begin
   opcode = OP_RDID;
   Twl_sck_i = Twl_sck_rd1;
   Twh_sck_i = Twh_sck_rd1;

   SPISendByte(opcode, 1, 0, Twl_sck_i, Twh_sck_i);
   SPIReadByte(rdidbuf[0], 0, Twl_sck_i, Twh_sck_i);
   SPIReadByte(rdidbuf[1], 1, Twl_sck_i, Twh_sck_i);
end
endtask

// 15 Reset device via SPI.
task CMD_Reset;
time Twl_sck_i;
time Twh_sck_i;
begin
   opcode = OP_Reset;
   Twl_sck_i = Twl_sck_min;
   Twh_sck_i = Twh_sck_min;

   SPISendByte(opcode, 1, 0, Twl_sck_i, Twh_sck_i);
   SPISendByte(8'hD0, 0, 1, Twl_sck_i, Twh_sck_i);
   #(Twh1_ce - Twh2_ce);
end
endtask

endmodule